/*
 * Decompiled with CFR 0.152.
 */
package com.fathzer.jchess.uci;

import com.fathzer.jchess.uci.BackgroundTaskManager;
import com.fathzer.jchess.uci.ConsoleLineReader;
import com.fathzer.jchess.uci.Engine;
import com.fathzer.jchess.uci.GoReply;
import com.fathzer.jchess.uci.StoppableTask;
import com.fathzer.jchess.uci.ThrowingRunnable;
import com.fathzer.jchess.uci.UCIMove;
import com.fathzer.jchess.uci.option.Option;
import com.fathzer.jchess.uci.parameters.GoParameters;
import com.fathzer.jchess.uci.parameters.Parser;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;

public class UCI
implements Runnable,
AutoCloseable {
    public static final String INIT_COMMANDS_PROPERTY_FILE = "uciInitCommands";
    private static final String MOVES = "moves";
    private static final String ENGINE_CMD = "engine";
    private static final String GO_CMD = "go";
    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.nnnnnnnn");
    private Supplier<String> in;
    protected Engine engine;
    private final Map<String, Consumer<Deque<String>>> executors = new HashMap<String, Consumer<Deque<String>>>();
    private final Map<String, Engine> engines = new HashMap<String, Engine>();
    private final BackgroundTaskManager backTasks = new BackgroundTaskManager();
    private boolean debug = Boolean.getBoolean("logToFile");
    private boolean debugUCI = Boolean.getBoolean("debugUCI");
    private Map<String, Option<?>> options;
    private boolean isPositionSet;
    private boolean isRunning;

    public UCI(Engine defaultEngine) {
        this.engines.put(defaultEngine.getId(), defaultEngine);
        this.engine = defaultEngine;
        this.addCommand(this::doUCI, "uci", new String[0]);
        this.addCommand(this::doDebug, "debug", new String[0]);
        this.addCommand(this::doSetOption, "setoption", new String[0]);
        this.addCommand(this::doIsReady, "isready", new String[0]);
        this.addCommand(this::doNewGame, "ucinewgame", "ng");
        this.addCommand(this::doPosition, "position", new String[0]);
        this.addCommand(this::doGo, GO_CMD, new String[0]);
        this.addCommand(this::doStop, "stop", new String[0]);
        this.addCommand(this::doEngine, ENGINE_CMD, new String[0]);
        this.addCommand(this::doQuit, "quit", "q");
    }

    public void add(Engine engine) {
        String id = engine.getId();
        if (id == null || id.isBlank()) {
            throw new IllegalArgumentException("Engine can't have a null or blank id");
        }
        if (this.engines.containsKey(id)) {
            throw new IllegalArgumentException("There's already an engine with id " + id);
        }
        this.engines.put(id, engine);
    }

    public Engine removeEngine(String id) {
        if (id.equals(this.engine.getId())) {
            throw new IllegalStateException("Can't remove current engine");
        }
        return this.engines.remove(id);
    }

    protected void addCommand(Consumer<Deque<String>> method, String command, String ... aliases) {
        if (command == null || command.isBlank() || method == null) {
            throw new IllegalArgumentException();
        }
        if (Arrays.stream(aliases).anyMatch(c -> c == null || c.isBlank())) {
            throw new IllegalArgumentException();
        }
        this.executors.put(command, method);
        Arrays.stream(aliases).forEach(c -> this.executors.put((String)c, method));
    }

    protected void doDebug(Deque<String> tokens) {
        if (tokens.size() == 1) {
            String arg = tokens.pop();
            if ("on".equals(arg)) {
                this.debugUCI = true;
            } else if ("off".equals(arg)) {
                this.debugUCI = false;
            } else {
                this.debug("Wrong argument " + arg);
            }
        } else {
            this.debug("Expected 1 argument to this command");
        }
    }

    protected void doUCI(Deque<String> tokens) {
        this.out("id name " + this.engine.getId());
        String author = this.engine.getAuthor();
        if (author != null) {
            this.out("id author " + author);
        }
        this.getOptions().values().forEach(o -> this.out(o.toUCI()));
        this.out("uciok");
    }

    private String processOption(Deque<String> tokens) {
        if (tokens.size() < 2) {
            return "Missing name prefix or option name";
        }
        if (!"name".equals(tokens.peek())) {
            return "setoption command should start with name";
        }
        String name = tokens.stream().skip(1L).takeWhile(t -> !"value".equals(t)).collect(Collectors.joining(" "));
        String value = tokens.stream().dropWhile(t -> !"value".equals(t)).skip(1L).collect(Collectors.joining(" "));
        if (name.isEmpty()) {
            return "Option name is empty";
        }
        Option<?> option = this.getOptions().get(name);
        if (option == null) {
            return "Unknown option";
        }
        try {
            option.setValue(value.isEmpty() ? null : value);
            return null;
        }
        catch (IllegalArgumentException e) {
            return "Value " + value + " is illegal";
        }
    }

    protected void doSetOption(Deque<String> tokens) {
        String error = this.processOption(tokens);
        if (error != null) {
            this.debug(error);
        }
    }

    protected void doIsReady(Deque<String> tokens) {
        this.out("readyok");
    }

    protected void doNewGame(Deque<String> tokens) {
        this.engine.newGame();
        this.isPositionSet = false;
    }

    protected void doPosition(Deque<String> tokens) {
        String fen;
        if (tokens.isEmpty()) {
            this.debug("missing position definition");
            return;
        }
        String first = tokens.pop();
        if ("fen".equals(first)) {
            fen = this.getFEN(tokens);
        } else if ("startpos".equals(first)) {
            fen = "rnbqkbnr/pppppppp/8/8/8/8/PPPPPPPP/RNBQKBNR w KQkq - 0 1";
        } else {
            this.debug("invalid position definition");
            return;
        }
        this.log("Setting board to FEN", fen);
        try {
            this.engine.setStartPosition(fen);
            tokens.stream().dropWhile(t -> !MOVES.equals(t)).skip(1L).forEach(this::doMove);
            this.isPositionSet = true;
        }
        catch (IllegalArgumentException e) {
            this.debug("invalid position definition");
        }
    }

    private void doMove(String move) {
        this.log("Moving", move);
        try {
            this.engine.move(UCIMove.from(move));
        }
        catch (IllegalArgumentException e) {
            this.debug("invalid move " + move);
        }
    }

    private String getFEN(Collection<String> tokens) {
        return tokens.stream().takeWhile(t -> !MOVES.equals(t)).collect(Collectors.joining(" "));
    }

    protected boolean doBackground(ThrowingRunnable task, Runnable stopper, Consumer<Exception> logger) {
        return this.backTasks.doBackground(new BackgroundTaskManager.Task(task, stopper, logger));
    }

    protected void doGo(Deque<String> tokens) {
        if (!this.isPositionSet()) {
            this.debug("No position defined");
        } else {
            Optional<GoParameters> goOptions = this.parse(GoParameters::new, GoParameters.PARSER, tokens);
            if (goOptions.isPresent()) {
                StoppableTask<GoReply> task = this.engine.go(goOptions.get());
                boolean started = this.doBackground(() -> this.processGo(task), task::stop, e -> this.err(GO_CMD, (Throwable)e));
                if (!started) {
                    this.debug("Engine is already working");
                }
            }
        }
    }

    private void processGo(StoppableTask<GoReply> task) throws Exception {
        GoReply goReply = (GoReply)task.call();
        Optional<String> mainInfo = goReply.getMainInfoString();
        if (mainInfo.isPresent()) {
            this.out(mainInfo.get());
            Optional<GoReply.Info> info = goReply.getInfo();
            int nb = info.isPresent() ? info.get().getExtraMoves().size() : 0;
            for (int i = 1; i <= nb; ++i) {
                goReply.getInfoString(i).ifPresent(this::out);
            }
        }
        this.out(goReply.toString());
    }

    protected <T> Optional<T> parse(Supplier<T> builder, Parser<T> parser, Deque<String> tokens) {
        try {
            T result = builder.get();
            List<String> ignored = parser.parse(result, tokens);
            if (!ignored.isEmpty()) {
                this.debug("The following parameters were ignored " + ignored);
            }
            return Optional.of(result);
        }
        catch (IllegalArgumentException e) {
            this.debug("There's an illegal argument in " + tokens);
            return Optional.empty();
        }
    }

    protected void doStop(Deque<String> tokens) {
        if (!this.backTasks.stop()) {
            this.debug("Nothing to stop");
        }
    }

    protected void doEngine(Deque<String> tokens) {
        if (tokens.isEmpty()) {
            this.out("engine " + this.engine.getId());
            this.engines.keySet().stream().filter(engineId -> !engineId.equals(this.engine.getId())).forEach(engineId -> this.out("engine " + engineId));
            return;
        }
        String engineId2 = tokens.peek();
        Engine newEngine = this.engines.get(engineId2);
        if (newEngine != null) {
            if (newEngine.equals(this.engine)) {
                return;
            }
            if (this.isPositionSet()) {
                this.isPositionSet = false;
                this.debug("position is cleared by engine change");
            }
            this.engine = newEngine;
            this.options = null;
            this.out("engine " + engineId2 + " ok");
        } else {
            this.debug("engine " + engineId2 + " is unknown");
        }
    }

    private Map<String, Option<?>> getOptions() {
        if (this.options == null) {
            this.options = this.engine.getOptions();
        }
        return this.options;
    }

    protected void doQuit(Deque<String> tokens) {
        this.isRunning = false;
    }

    @Override
    public void run() {
        this.init();
        this.isRunning = true;
        while (this.isRunning) {
            this.log("Waiting for command...");
            String command = this.getNextCommand().trim();
            if (command.isEmpty()) continue;
            this.doCommand(command);
        }
    }

    private void init() {
        String initFile = System.getProperty(INIT_COMMANDS_PROPERTY_FILE);
        if (initFile != null) {
            try {
                Files.readAllLines(Paths.get(initFile, new String[0])).stream().map(String::trim).filter(s -> !s.isEmpty()).forEach(this::doCommand);
            }
            catch (IOException e) {
                this.err("init engine", e);
            }
        }
    }

    protected boolean doCommand(String command) {
        this.log(">", command);
        LinkedList<String> tokens = new LinkedList<String>(Arrays.asList(command.split(" ")));
        Consumer<Deque<String>> executor = this.executors.get(tokens.pop());
        if (executor == null) {
            this.debug("unknown command");
            return false;
        }
        try {
            executor.accept(tokens);
        }
        catch (RuntimeException e) {
            this.err(command, e);
        }
        return true;
    }

    protected void err(String tag, Throwable e) {
        this.err("Error with " + tag + " tag");
        this.err(e, 0);
    }

    private void err(Throwable e, int level) {
        this.err((level > 0 ? "caused by " : "") + e.toString());
        Arrays.stream(e.getStackTrace()).forEach(f -> this.err(f.toString()));
        if (e.getCause() != null) {
            this.err(e.getCause(), level + 1);
        }
    }

    protected void err(CharSequence message) {
        System.err.println(message);
    }

    private void log(String ... message) {
        this.log(true, message);
    }

    private synchronized void log(boolean append, String ... messages) {
        if (!this.debug) {
            return;
        }
        try (BufferedWriter out = new BufferedWriter(new FileWriter("log.txt", append));){
            out.write(LocalDateTime.now().format(DATE_FORMAT));
            out.write(" - ");
            for (String mess : messages) {
                out.write(mess);
                out.write(32);
            }
            out.newLine();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    protected Supplier<String> getInputSupplier() {
        if (this.in == null) {
            this.in = new ConsoleLineReader();
        }
        return this.in;
    }

    protected String getNextCommand() {
        return this.getInputSupplier().get().trim();
    }

    protected void out(CharSequence message) {
        this.log(":", message.toString());
        System.out.println(message);
    }

    protected boolean isDebugMode() {
        return this.debugUCI;
    }

    protected void debug(CharSequence message) {
        this.log(":", "info", "UCI debug is", Boolean.toString(this.debugUCI), message.toString());
        if (this.debugUCI) {
            this.out("info string " + message);
        }
    }

    protected boolean isPositionSet() {
        return this.isPositionSet;
    }

    @Override
    public void close() {
        this.backTasks.close();
    }
}

